/*
 *
 * maps.c
 * Created: 20090106 Jerome Hert
 *  
 */

#include "maps.h"

/*
 * +--------------------------------------------------------------------------+
 * | Function for Map                                                         |
 * +--------------------------------------------------------------------------+
 */

t_fp *map_initialize(t_fp *map) {
	int i,j;
	for (i=ZERO;i<ICOSA_NUM_TRIANGLES;i++)
		for (j=ZERO;j<FP_SIZE;j++)
			map[i][j] = ZERO;
	return map;
}

t_fp *map_copy(t_fp *to, t_fp *from) {
	int i,j;
	for (i=ZERO;i<ICOSA_NUM_TRIANGLES;i++)
		for (j=ZERO;j<FP_SIZE;j++)
			to[i][j] = from[i][j];
	return to;
}

t_fp *map_compute(t_fp *map, t_protein *protein) {

	int i, icosa_triangle, assign_residue;
	t_residue *residue = NULL, *residue2 = NULL;

	if (DEBUG_MAP) {
		fprintf(stderr, "WARNING:\t-------------------------------------------------------------\n");
		fprintf(stderr, "WARNING:\tTriangle assignment for file %s\n", protein->file_protein);
	}

	/* Attribute a triangle to each residue */
	for(i=ZERO;i<protein->size;i++) {
		/* Initialize */
		icosa_triangle  = MINUSONE;
		assign_residue  = TRUE;

		residue = &protein->residue[i];

		/* Skip if unknown residue type */
		if (residue->type == '?')
			continue;

		/*
		 * The Triangle is assigned based on the projection of
		 * the line formed by the origin and the Cbeta.
		 * In the case of a Glycine, the Calpha is used instead of
		 * the Cbeta, but Cb in the residue struct already points
		 * to the Calpha (so we do nothing)
		 */
		icosa_triangle = mapTriangle(&residue->atom[residue->Cb].coord);

		/*
		 * From this point only continue if the returned value
		 * corresponds to one of the triangles of the icosahedre
		 */
		if (icosa_triangle >= ZERO && icosa_triangle < ICOSA_NUM_TRIANGLES) {
			/*
			 * Check if a residue has already been assigned to this triangle
			 * If it is the case look if the distance of the Cbeta to the origin
			 * of the residue currently being considered is shorter than the distance
			 * of the Cbeta to the origin of the residue already assigned.
			 * If it is shorter, changes the residue, otherwise just ignore it.
			 * Rem: if the residue is a glycin, it is the distance to the Calpha
			 * that is computed.
			 */
			if (DEBUG_MAP)
				fprintf(stderr, "WARNING:\tResidue [%c]: %c %d - Triangle: %d\n", residue->type, residue->chain, residue->id, icosa_triangle);

			if (map[icosa_triangle][FP_BITS] == TRUE) {
				residue2 = &protein->residue[ map[icosa_triangle][FP_RESIDUE_INDEX] ];
				if ( point_distance_origin(&residue2->atom[residue2->Cb].coord) < point_distance_origin(&residue->atom[residue->Cb].coord) ) {
					assign_residue = FALSE;
				}
				if (DEBUG_MAP)
					fprintf(stderr, "WARNING:\tTriangle %d - Residue [%c %c %d] %8.3f - Residue [%c %c %d] %8.3f - Assign: %d\n",
						icosa_triangle,
						residue->type, residue->chain, residue->id, point_distance_origin(&residue2->atom[residue2->Cb].coord),
						protein->residue[i].type, protein->residue[i].chain, protein->residue[i].id, point_distance_origin(&residue->atom[residue->Cb].coord),
						assign_residue);
			}
			
			/* Assign residue */
			if (assign_residue) {
				map[icosa_triangle][FP_RESIDUE_INDEX]	= i;
				map[icosa_triangle][FP_BITS]			= TRUE;
			}
		}
	}

	/* compute descriptors for mapping */
	map_compute_fp(map, protein);

	return map;
}

t_fp *map_compute_fp(t_fp *map, t_protein *protein) {
	int i;
	for (i=ZERO;i<ICOSA_NUM_TRIANGLES;i++) {
		if (map[i][FP_BITS] == TRUE)
			fp_compute_fp( &map[i], &protein->residue[ map[i][FP_RESIDUE_INDEX] ], &protein->centerofca );
	}
	if (DEBUG_MAP) {
		fprintf(stderr, "WARNING:\t-------------------------------------------------------------\n");
		fprintf(stderr, "WARNING:\tMAPPING for file %s\n", protein->file_protein);
		map_print(stderr, map );
	}
	return map;
}

t_score *map_compare(t_score *score, t_fp *map1, t_fp *map2, int score_d3) {
	int i, N = ZERO, N2 = ZERO, N3 = ZERO, N4 = ZERO;
	double score_tmp = ZERO, score_tot_tmp = ZERO;
	for (i=ZERO;i<ICOSA_NUM_TRIANGLES;i++) {
		if ( map1[i][FP_BITS] == TRUE || map2[i][FP_BITS] == TRUE )
			N++;
		if ( map1[i][FP_BITS] == TRUE && map2[i][FP_BITS] == TRUE ) {
			N2++;
			score_tmp = fp_compare(&map1[i], &map2[i]);
			score_tot_tmp += score_tmp;
			if (score_tmp >= SCORE_N3_THRESHOLD) {
				N3++; /* Number of matching residue */
				if ( ( map1[i][FP_DONOR] != ZERO || map1[i][FP_ACCEPTOR] != ZERO ) && ( map2[i][FP_DONOR] != ZERO || map2[i][FP_ACCEPTOR] != ZERO) )
					N4++; /* Number of polar matching residue */
			}
		}
	}
	if (score_d3 == ZERO) {
		score->d1	= N  == ZERO ? ONE : ONE - score_tot_tmp / N;
		score->d2	= N2 == ZERO ? ONE : ONE - score_tot_tmp / N2;
		score->N	= N;
		score->N2	= N2;
		score->N3	= N3;
		score->N4	= N4;
	}
	else {
		score->d3	= N2 == ZERO ? ONE : ONE - score_tot_tmp / N2;
	}
	return score;
}

t_score *map_calculate_d4(t_score *score, t_fp *map1, t_fp *map2, t_protein *p1, t_protein *p2) {
	int i, index1, index2, count = ZERO;
	double rmsd = ZERO;
	t_residue *res1, *res2;
	for (i=ZERO;i<ICOSA_NUM_TRIANGLES;i++) {
		if ( map1[i][FP_BITS] == TRUE && map2[i][FP_BITS] == TRUE ) {
			index1	= map1[i][FP_RESIDUE_INDEX];
			index2	= map2[i][FP_RESIDUE_INDEX];
			res1	= &p1->residue[index1];
			res2	= &p2->residue[index2];
			index1	= res1->Cb;
			index2	= res2->Cb;
			rmsd	+= square( point_distance(&res1->atom[index1].coord, &res2->atom[index2].coord) );
			count++;
		}
	}
	score->d4 = sqrt( rmsd / count );
	return score;
}

void map_print(FILE *fh, t_fp *map) {
	int i,j;
	for (i=ZERO;i<ICOSA_NUM_TRIANGLES;i++) {
		fprintf(fh, "WARNING:\t");
		for (j=ZERO;j<FP_SIZE-1;j++) {
			fprintf(fh, "%d ", map[i][j]);
		}
		fprintf(fh, "%d\n", map[i][FP_SIZE-1]);
	}
	return;
}

/*
 * +--------------------------------------------------------------------------+
 * | Function for Fp                                                          |
 * +--------------------------------------------------------------------------+
 */

t_fp *fp_compute_fp(t_fp *fp, t_residue *residue, t_point *centerofca) {

	double distCa = 0., distCb = 0., distCenterOfMass = 0.;

	/*
	 * Calculate the distance to the origin
	 * Here point_tmp is always the corresponding point
	 * when the center of the Calpha of the site is moved to the origin 0, 0, 0 
	 */

	/* Distance du Cb a l'origine */
	distCb				= point_distance_origin(&residue->atom[residue->Cb].coord);
	(*fp)[FP_DIST]		= distance_discretize(distCb);
	
	/* Orientation de la chaine */
	distCa				= point_distance_origin(&residue->atom[residue->Ca].coord);
	distCenterOfMass	= point_distance_origin(&residue->centerofmass);
	(*fp)[FP_SCOR]		= distCenterOfMass < distCa ? INSIDE : OUTSIDE;

	/* Assign the rest of the descritors based on the residue type */
	switch (residue->type) {

		case 'A':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'R':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	THREE;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	THREE;
			(*fp)[FP_CHARGE]	=	ONE;
			break;

		case 'N':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ONE;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'D':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	TWO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	MINUSONE;
			break;

		case 'C':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'Q':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ONE;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'E':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	TWO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	MINUSONE;
			break;

		case 'H':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ONE;
			(*fp)[FP_AROMATIC]	=	ONE;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'J':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	TWO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ONE;
			break;

		case 'I':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'L':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'K':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ONE;
			break;

		case 'M':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	TWO;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'F':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ONE;
			(*fp)[FP_SCSIZE]	=	THREE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'P':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'S':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ONE;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'T':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ONE;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'W':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ONE;
			(*fp)[FP_SCSIZE]	=	THREE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'Y':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ONE;
			(*fp)[FP_ACCEPTOR]	=	ONE;
			(*fp)[FP_AROMATIC]	=	ONE;
			(*fp)[FP_SCSIZE]	=	THREE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'V':
			(*fp)[FP_ALIPHATIC]	=	ONE;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case 'G':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ONE;
			(*fp)[FP_CHARGE]	=	ZERO;
			break;

		case '?':
			(*fp)[FP_ALIPHATIC]	=	ZERO;
			(*fp)[FP_DONOR]		=	ZERO;
			(*fp)[FP_ACCEPTOR]	=	ZERO;
			(*fp)[FP_AROMATIC]	=	ZERO;
			(*fp)[FP_SCSIZE]	=	ZERO;
			(*fp)[FP_CHARGE]	=	ZERO;
			(*fp)[FP_DIST]      =   ZERO;
			(*fp)[FP_SCOR]      =   ZERO;

		default:
			fprintf(stderr, "WARNING: unknown residue type (%s) %c %d. Setting all values of its FP to 0.\n",residue->type_long, residue->chain, residue->id);
	}

	return fp;
}

double fp_compare(t_fp *fp1, t_fp *fp2) {
	double score_tmp = 0., double_tmp = 0.;
	/* Aliphatic Part */
	score_tmp	+= ONE - fabs( (*fp1)[FP_ALIPHATIC] - (*fp2)[FP_ALIPHATIC] );
	/* Donor Part */
	score_tmp	+= ONE - fabs( (*fp1)[FP_DONOR] - (*fp2)[FP_DONOR] ) / THREE;
	/* Acceptor Part */
	score_tmp	+= ONE - fabs( (*fp1)[FP_ACCEPTOR] - (*fp2)[FP_ACCEPTOR] ) / TWO;
	/* Aromatic Part */
	score_tmp	+= ONE - fabs( (*fp1)[FP_AROMATIC] - (*fp2)[FP_AROMATIC] );
	/* Distance Part */
	double_tmp	=  fabs( (*fp1)[FP_DIST] - (*fp2)[FP_DIST] );
	score_tmp	+= double_tmp > THIRTY ? ZERO : ONE - double_tmp / THIRTY;
	/* Side Chain Size Part */
	score_tmp	+= ONE - fabs( (*fp1)[FP_SCSIZE] - (*fp2)[FP_SCSIZE] ) / TWO;
	/* Side Chain Orientation Part */
	score_tmp	+= ONE - fabs( (*fp1)[FP_SCOR] - (*fp2)[FP_SCOR] );
	/* Charge Part */
	double_tmp	=  fabs( (*fp1)[FP_CHARGE] - (*fp2)[FP_CHARGE] );
	score_tmp	+= double_tmp > TWO ? ZERO : ONE - double_tmp / TWO;
	return score_tmp / FP_SIZE_SCORE;
}
