/*
 *
 * align.c
 * Created: 20090106 Jerome Hert
 *  
 */

#include "align.h"

/*
 * +--------------------------------------------------------------------------+
 * | Function for Alignement Structures                                       |
 * +--------------------------------------------------------------------------+
 */

t_alignment *alignment_initialize(t_alignment *alignment) {
	transfo_initialize(&alignment->transfo);
	map_initialize(alignment->map);
	score_initialize(&alignment->score);
	return alignment;
}

t_alignment *alignment_copy(t_alignment *to, t_alignment *from) {
	transfo_copy(&to->transfo, &from->transfo);
	map_copy(to->map, from->map);
	score_copy(&to->score, &from->score);
	return to;
}

/*
 * +--------------------------------------------------------------------------+
 * | Alignment functions                                                      |
 * +--------------------------------------------------------------------------+
 */

t_alignment *align( t_protein *p_query,	t_protein *p_target, t_alignment *alignment_query, t_alignment *alignment_target, t_explore *explore_params, int nb_passes, int verbosity) {

	int i;
	t_alignment	n_best_alignments[nb_passes];
	t_explore	refine_params;

	/* Calculate Refinement Parameters */
	explore_copy(&refine_params, explore_params);
	explore_calculate_refinement(&refine_params);

	/* Initialize all the alignments */
	for (i=ZERO;i<nb_passes;i++)
		alignment_initialize(&n_best_alignments[i]);

	if (verbosity) {
		fprintf( stderr, "WARNING:\t-------------------------------------------------------------\n");
		fprintf( stderr, "WARNING:\tGlobal search:  Finding the best %d solutions for %s\n", nb_passes, p_target->file_protein);
		fprintf( stderr, "WARNING:\tExploration parameters: " );
		explore_print(stderr, explore_params);
		fprintf( stderr, "WARNING:\t-------------------------------------------------------------\n");
	}

	/* Align original protein with original parameters and keep 3 best results */
	align_n_passes( alignment_query->map, p_target, n_best_alignments, explore_params, nb_passes);

	if (verbosity) {
		for (i=ZERO;i<nb_passes;i++) {
			fprintf(stderr, "WARNING:\tSolution %d ->", i+1);
			fprintf(stderr, " [d1=%.5f d2=%.5f]", n_best_alignments[i].score.d1, n_best_alignments[i].score.d2);
			fprintf(stderr, " (alpha=%.2f beta=%.2f gamma=%.2f)", n_best_alignments[i].transfo.rotation_angle.alpha * 180 / PI, n_best_alignments[i].transfo.rotation_angle.beta * 180 / PI, n_best_alignments[i].transfo.rotation_angle.gamma * 180 / PI);
			fprintf(stderr, " (vx=%.2f vy=%.2f vz=%.2f)\n", n_best_alignments[i].transfo.translation.x, n_best_alignments[i].transfo.translation.y, n_best_alignments[i].transfo.translation.z);
		}
		fprintf( stderr, "WARNING:\t-------------------------------------------------------------\n");
		fprintf( stderr, "WARNING:\tRefinement:  Local search around the best %d solutions\n", nb_passes);
		fprintf( stderr, "WARNING:\tExploration parameters: " );
		explore_print(stderr, &refine_params);
		fprintf( stderr, "WARNING:\t-------------------------------------------------------------\n");
	}

	/* Optimize each pass */
	if (n_best_alignments[ZERO].score.d1 != ZERO) {
		for (i=ZERO;i<nb_passes;i++) {
			if (verbosity) {
				fprintf(stderr, "WARNING:\tRefining Solution %d ->", i+1);
			}
			align_optimize( alignment_query->map, p_target, &n_best_alignments[i] ,&refine_params);
			if (verbosity) {
				fprintf(stderr, " [d1=%.5f d2=%.5f]", n_best_alignments[i].score.d1, n_best_alignments[i].score.d2);
				fprintf(stderr, " (alpha=%.2f beta=%.2f gamma=%.2f)", n_best_alignments[i].transfo.rotation_angle.alpha * 180 / PI, n_best_alignments[i].transfo.rotation_angle.beta * 180 / PI, n_best_alignments[i].transfo.rotation_angle.gamma * 180 / PI);
				fprintf(stderr, " (vx=%.2f vy=%.2f vz=%.2f)\n", n_best_alignments[i].transfo.translation.x, n_best_alignments[i].transfo.translation.y, n_best_alignments[i].transfo.translation.z);
			}
			if (n_best_alignments[i].score.d1 == ZERO)
				break;
		}
	}

	/* Find the best solution, copy it and exit */
	i = alignment_get_index_best(n_best_alignments, nb_passes);
	alignment_copy(alignment_target, &n_best_alignments[i]);
	calculate_d3_d4(p_query, p_target, alignment_query, alignment_target);
	
	return alignment_target;
}

t_alignment *calculate_d3_d4(t_protein *p_query, t_protein *p_target, t_alignment *alignment_query, t_alignment *alignment_target) {
	t_protein *pq = NULL, *pt = NULL;
	t_fp map_query[ICOSA_NUM_TRIANGLES], map_target[ICOSA_NUM_TRIANGLES];
	/* Copy proteins and align all atoms to new position */
	pq = (t_protein *)calloc(ONE, sizeof(t_protein));
	pt = (t_protein *)calloc(ONE, sizeof(t_protein));
	protein_copy(pq, p_query);
	protein_copy(pt, p_target);
	/*
	 * Get the rotamer specific side chain atom for each residue
	 * Its index is stored in protein->residue[i].Cb
	 * That's why we need to worked on a copy of the protein (especially for the query)
	 */
	if ( protein_prepare_d3_d4(pq) != ZERO )
		fprintf(stderr, "WARNING:\tThere were missing rotamer specific side chain atoms in file \"%s\".\n", pq->file_protein);
	if ( protein_prepare_d3_d4(pt) != ZERO )
		fprintf(stderr, "WARNING:\tThere were missing rotamer specific side chain atoms in file \"%s\".\n", pt->file_protein);
	/* Rotate the protein CA CB and center of mass, here CB is the rotamer spcific side chain atom though */
	protein_transform_ca_cb_cm(pq, &alignment_query->transfo);
	protein_transform_ca_cb_cm(pt, &alignment_target->transfo);
	/* Copy maps and then recalculate the FP for each assigned triangle */
	map_copy(map_query,	alignment_query->map);
	map_copy(map_target,alignment_target->map);
	map_compute_fp(map_query, pq);
	map_compute_fp(map_target, pt);
	/* Calculate d3 */
	map_compare(&alignment_target->score, map_query, map_target, ONE);
	/* Calculate d4 */
	map_calculate_d4(&alignment_target->score, map_query, map_target, pq, pt);
	/* Clean up */
	protein_free(pq);
	protein_free(pt);
	free(pq);
	free(pt);	
	return alignment_target;
}

t_alignment *align_n_passes( t_fp *map, t_protein *protein, t_alignment *alignments, t_explore *explore_params, int nb_passes) {

	double alpha, beta, gamma, vx, vy, vz;
	t_point		point_tmp;
	t_angle		angle_tmp;
	t_alignment	alignment_tmp;
	t_protein	protein_tmp;
	
	protein_initialize(&protein_tmp);
	protein_copy(&protein_tmp, protein);

	for(vx = -explore_params->translation_int / TWO; vx <= explore_params->translation_int / TWO; vx += explore_params->translation_increment)

		for(vy = -explore_params->translation_int / TWO; vy <= explore_params->translation_int / TWO; vy += explore_params->translation_increment)

			for(vz = -explore_params->translation_int / TWO; vz <= explore_params->translation_int / TWO; vz += explore_params->translation_increment)

				for (alpha = -explore_params->rotation_int / TWO; alpha <= explore_params->rotation_int / TWO; alpha += explore_params->rotation_increment)

					for (beta = -explore_params->rotation_int / TWO; beta <= explore_params->rotation_int / TWO; beta += explore_params->rotation_increment)

						for (gamma = -explore_params->rotation_int / TWO; gamma <= explore_params->rotation_int / TWO; gamma += explore_params->rotation_increment) {

							point_set(&point_tmp, vx, vy, vz);
							angle_set(&angle_tmp, alpha, beta, gamma);
							transfo_set(&alignment_tmp.transfo, angle_tmp, point_tmp);
							protein_copy_ca_cb_cm(&protein_tmp, protein);
							protein_transform_ca_cb_cm(&protein_tmp, &alignment_tmp.transfo);
							map_initialize(alignment_tmp.map);
							map_compute(alignment_tmp.map, &protein_tmp);
							score_initialize(&alignment_tmp.score);
							map_compare(&alignment_tmp.score, map, alignment_tmp.map, ZERO);
							keep_best_alignement(alignments, &alignment_tmp, nb_passes);
							if (alignments[ZERO].score.d1 == ZERO) {
								protein_free(&protein_tmp);
								return alignments;
							}
//							if (DEBUG_ALIGN) {
//								fprintf(stderr, "WARNING:\t-------------------------------------------------------------\n");
//								fprintf(stderr, "WARNING:\tPARAMS -> a: %g - b: %g - g: %g - vx: %g - vy: %g - vz: %g\n", alpha,beta,gamma,vx,vy,vz);
//								fprintf(stderr, "WARNING:\tSCORES -> tmp[%g - %g], stack1[%g - %g], stack2[%g - %g], stack3[%g - %g]\n", 
//									alignment_tmp.score.d1, alignment_tmp.score.d2,
//									(&alignments[ZERO])->score.d1, (&alignments[ZERO])->score.d2,
//									(&alignments[ONE])->score.d1,  (&alignments[ONE])->score.d2,
//									(&alignments[TWO])->score.d1,  (&alignments[TWO])->score.d2
//								);
//							}
						}

	protein_free(&protein_tmp);
	return alignments;
}

t_alignment *align_optimize( t_fp *map, t_protein *protein, t_alignment *alignment, t_explore *explore_params) {
	double alpha, beta, gamma, vx, vy, vz;
	t_point		point_tmp;
	t_angle		angle_tmp;
	t_alignment	alignment_tmp, alignment_best;
	t_protein	protein_tmp;

	alignment_copy(&alignment_best, alignment);

	protein_initialize(&protein_tmp);
	protein_copy(&protein_tmp, protein);

	for(vx = alignment->transfo.translation.x - explore_params->translation_int / TWO; vx <= alignment->transfo.translation.x + explore_params->translation_int / TWO; vx += explore_params->translation_increment)

		for(vy = alignment->transfo.translation.y - explore_params->translation_int / TWO; vy <= alignment->transfo.translation.y + explore_params->translation_int / TWO; vy += explore_params->translation_increment)

			for(vz = alignment->transfo.translation.z - explore_params->translation_int / TWO; vz <= alignment->transfo.translation.z + explore_params->translation_int / TWO; vz += explore_params->translation_increment)

				for (alpha = alignment->transfo.rotation_angle.alpha - explore_params->rotation_int / TWO; alpha <= alignment->transfo.rotation_angle.alpha + explore_params->rotation_int / TWO; alpha += explore_params->rotation_increment)

					for (beta = alignment->transfo.rotation_angle.beta - explore_params->rotation_int / TWO; beta <= alignment->transfo.rotation_angle.beta + explore_params->rotation_int / TWO; beta += explore_params->rotation_increment)

						for (gamma = alignment->transfo.rotation_angle.gamma - explore_params->rotation_int / TWO; gamma <= alignment->transfo.rotation_angle.gamma + explore_params->rotation_int / TWO; gamma += explore_params->rotation_increment) {

							point_set(&point_tmp, vx, vy, vz);
							angle_set(&angle_tmp, alpha, beta, gamma);
							transfo_set(&alignment_tmp.transfo, angle_tmp, point_tmp);
							protein_copy_ca_cb_cm(&protein_tmp, protein);
							protein_transform_ca_cb_cm(&protein_tmp, &alignment_tmp.transfo);
							map_initialize(alignment_tmp.map);
							map_compute(alignment_tmp.map, &protein_tmp);
							score_initialize(&alignment_tmp.score);
							map_compare(&alignment_tmp.score, map, alignment_tmp.map, ZERO);
							if (alignment_tmp.score.d1 < alignment_best.score.d1) {
//								if (DEBUG_ALIGN) {
//									fprintf(stderr, "WARNING:\t-------------------------------------------------------------\n");
//									fprintf(stderr, "WARNING:\tPARAMS -> a: %g - b: %g - g: %g - vx: %g - vy: %g - vz: %g\n", alpha,beta,gamma,vx,vy,vz);
//									fprintf(stderr, "WARNING:\tSCORES -> tmp[%g - %g], stack[%g - %g]\n\n\n", 
//										alignment_tmp.score.d1, alignment_tmp.score.d2,
//										alignment_best.score.d1, alignment_best.score.d2
//									);
//								}
								alignment_copy(&alignment_best, &alignment_tmp);
							}
							if (alignment_best.score.d1 == ZERO) {
								alignment_copy(alignment, &alignment_best);
								return alignment;
							}
						}

	alignment_copy(alignment, &alignment_best);
	return alignment;
}

t_alignment *keep_best_alignement(t_alignment *alignments, t_alignment *alignment_current, int nb_passes) {
//	int i, j, index_same_zone;
	int i, j; 
	
	for (i=ZERO;i<nb_passes;i++) {
		if ( alignment_current->score.d1 < (&alignments[i])->score.d1 ) {
//			index_same_zone = MINUSONE;
//			for (j=ZERO;j<nb_passes;j++) {
//				if ( transfo_same_zone(&alignment_current->transfo, &(&alignments[i])->transfo) == TRUE ) {
//					index_same_zone = j;
//					break;
//				}
//			}
//			if (index_same_zone == i) {
//				alignment_copy(&alignments[i], alignment_current);
//			}
//			else if (index_same_zone == MINUSONE || index_same_zone > i) {
				for (j=nb_passes-TWO;j>=i;j--) {
					alignment_copy(&alignments[j+1], &alignments[j]);
				}
				alignment_copy(&alignments[i], alignment_current);
//			}
//			else if (index_same_zone == MINUSONE) {
//				for (j=nb_passes-TWO;j>=i;j--) {
//					alignment_copy(&alignments[j+1], &alignments[j]);
//				}
//				alignment_copy(&alignments[i], alignment_current);
//			}
//			else if (index_same_zone > i) {
//				for (j=index_same_zone-ONE;j>=i;j--) {
//					alignment_copy(&alignments[j+1], &alignments[j]);
//				}
//				alignment_copy(&alignments[i], alignment_current);
//			}
			break;
		}
	}

	return alignments;
}

int alignment_get_index_best(t_alignment *alignments, int size) {
	int i, index = ZERO;
	double best_score;
	best_score = (&alignments[ZERO])->score.d1;
	for (i=1; i<size; i++) {
		if ( best_score > (&alignments[i])->score.d1 ) {
			index = i;
			best_score = (&alignments[i])->score.d1;
		}
	}
	return index;
}
