package fr.u_strasbg.bioinfo_pharma.application.bestGoldScorer.processing;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import fr.u_strasbg.bioinfo_pharma.util.Short;
import fr.u_strasbg.bioinfo_pharma.application.bestGoldScorer.BestGoldScorer;
import fr.u_strasbg.bioinfo_pharma.application.bestGoldScorer.exception.BGSException;
import fr.u_strasbg.bioinfo_pharma.application.bestGoldScorer.ligand.Ligand;
import fr.u_strasbg.bioinfo_pharma.application.bestGoldScorer.ligand.Position;

public class Processor {

	//Constants
	public static final double NO_THRESHOLD = -9999.99;
	public static final int KEEP_NONE = -1;
	public static final int KEEP_ALL = 0;
	public static final int KEEP_BEST = 1;
	
	//Attributs
	private int keep;
	private int bundleSize;
	private double threshold;
	private String inputLigandFilePath;
	private String ouputScoreFilePath;
	private String outputMol2FilePath;
	private String baseDir;
	private String prefixDir;
	private String prefixSubDir;
	private String suffixSubDir;
	private String molFileRankExtension = ".rnk";
	private String fileSeparator = System.getProperty("file.separator");

	
	private int nbOfDirectories;
	private int nbOfLigands;
	private List<Ligand> ligands;
	private Position[] shortedPositions;
	private Position[] finalPositions;
	private int ligandId = 0;
	
	//Constructors
	public Processor(){
		this.setLigands( new ArrayList<Ligand>() );
		this.setKeep( KEEP_NONE );
	}
	
	//Getters
	public int getKeep(){
		return this.keep;
	}
	public int getBundleSize(){
		return this.bundleSize;
	}
	public double getThreshold(){
		return this.threshold;
	}
	public String getBaseDir(){
		return this.baseDir;
	}
	public String getPrefixDir(){
		return this.prefixDir;
	}
	public String getSubPrefixDir(){
		return this.prefixSubDir;
	}
	public String getSubSuffixDir(){
		return this.suffixSubDir;
	}
	public String getMolFileRankExtension(){
		return this.molFileRankExtension;
	}
	public String getInputLigandFilePath(){
		return this.inputLigandFilePath;
	}
	public String getOuputScoreFilePath(){
		return this.ouputScoreFilePath;
	}
	public String getOutputMol2FilePath(){
		return this.outputMol2FilePath;
	}
	
	//Setters
	public void setKeep( int keep ){
		this.keep = keep;
	}
	public void setBundleSize( int bundleSize ){
		this.bundleSize = bundleSize;
	}
	public void setThreshold( double threshold ){
		this.threshold = threshold;
	}
	public void setBaseDir( String baseDir ){
		this.baseDir = baseDir;
	}
	public void setPrefixDir( String prefixDir ){
		this.prefixDir = prefixDir;
	}
	public void setSubPrefixDir( String prefixSubDir ){
		this.prefixSubDir = prefixSubDir;
	}
	public void setSubSuffixDir( String suffixSubDir ){
		this.suffixSubDir = suffixSubDir;
	}
	public void setMolFileRankExtension( String molFileRankExtension ){
		this.molFileRankExtension = molFileRankExtension;
	}
	public void setInputLigandFilePath( String inputLigandFilePath ){
		this.inputLigandFilePath = inputLigandFilePath;
	}
	public void setOuputScoreFilePath( String ouputScoreFilePath ){
		this.ouputScoreFilePath = ouputScoreFilePath;
	}
	public void setOutputMol2FilePath( String outputMol2FilePath ){
		this.outputMol2FilePath = outputMol2FilePath;
	}
	private void setLigands( List<Ligand> ligands){
		this.ligands = ligands;
	}
	
	//Public method
	public void process () throws BGSException {
		BestGoldScorer.logger.info("Process is starting ...");
		this.init();
		
		BestGoldScorer.logger.info("Process is recovering poses ...");
		if ( this.keep == KEEP_ALL )
			this.visitDirectories( KEEP_ALL , this.nbOfDirectories , this.nbOfLigands );
		else 
			this.visitDirectories( KEEP_BEST , this.nbOfDirectories , this.nbOfLigands );
		
		BestGoldScorer.logger.info("Process is sorting data ...");
		this.shortByPositions();
		
		if ( this.keep == KEEP_ALL && this.getThreshold() != NO_THRESHOLD )
			this.finalPositions = this.cutOff();
		else if ( this.keep == KEEP_BEST && this.getThreshold() != NO_THRESHOLD )
			this.finalPositions = this.cutOff();
		else
			this.finalPositions = this.shortedPositions;
		
		this.concat();
		BestGoldScorer.logger.info("Process is finishing ...");
	}
	
	//Private init methods
	private void init() throws BGSException{
		this.checkAll();
		this.recoveryOfLigands();
		this.nbOfLigands = this.ligands.size() ;
		this.nbOfDirectories = this.getNbOfDirectories( this.nbOfLigands );
	}
	private void checkAll() throws BGSException{
		if ( this.getKeep() == KEEP_NONE )
			throw new BGSException("Option 'keep' has not a valid value.");
		if ( this.getBundleSize() < 1 )
			throw new BGSException("Option 'size' has not a valid value.");
		if ( this.getPrefixDir() == null || this.getPrefixDir().equals("") )
			throw new BGSException("Option 'prefixDir' has not a valid value.");
		if ( this.getSubPrefixDir() == null || this.getPrefixDir().equals("") )
			throw new BGSException("Option 'subPrefixDir' has not a valid value.");
		if ( this.getSubSuffixDir() == null || this.getPrefixDir().equals("") )
			throw new BGSException("Option 'subSuffixDir' has not a valid value.");
		if ( this.getMolFileRankExtension() == null ) {
			throw new BGSException("Option 'extension' has not a valid value.");
		}
		this.checkInputFile();
		this.checkOutputFiles();
	}
	private void checkInputFile() throws BGSException{
		if ( this.getInputLigandFilePath() == null || this.getInputLigandFilePath().equals("") )
			throw new BGSException("An input file must be given to the Processor class.");
		
		File file = new File( this.baseDir + this.inputLigandFilePath );
		if ( !file.exists() )
			throw new BGSException("This file named '"+this.baseDir + this.inputLigandFilePath+"' does not exist.");
		if ( !file.canRead() )
			throw new BGSException("You are not allowed to read the file named '"+this.baseDir + this.inputLigandFilePath+"'.");
	}
	private void checkOutputFiles() throws BGSException{
		if ( this.getOuputScoreFilePath() == null || this.getOuputScoreFilePath().equals("") )
			throw new BGSException("A score output file must be given to the Processor class.");
		//File file = new File( this.baseDir + this.getOuputScoreFilePath() );
		//if ( !file.canWrite() )
			//throw new BGSException("You are not allowed to write in the file named '"+this.getOuputScoreFilePath()+"'.");
		if ( this.getOutputMol2FilePath() == null || this.getOutputMol2FilePath().equals("") )
			throw new BGSException("A score output file must be given to the Processor class.");
		//file = new File( this.getOutputMol2FilePath() );
		//if ( !file.canWrite() )
			//throw new BGSException("You are not allowed to write in the file named '"+this.getOutputMol2FilePath()+"'.");
	}
	private void recoveryOfLigands() throws BGSException{
		try {
			File inputFile = new File ( this.baseDir + this.inputLigandFilePath );
			FileInputStream fis = new FileInputStream( inputFile );
			InputStreamReader isr = new InputStreamReader( fis );
			BufferedReader br = new BufferedReader( isr );
			
			String line = null;
			int ligNb = 0;
			while ( (line = br.readLine()) != null ){
				
				if ( !line.equals("") ){
					ligNb++;
					Ligand ligand = new Ligand();
					ligand.setNumber( ligNb );
					ligand.setName( line.trim() );
					ligand.setPositions( new ArrayList<Position>() );
					this.ligands.add( ligand );
				}
			}
			br.close();
			isr.close();
			fis.close();
		} catch ( FileNotFoundException e){
			throw new BGSException( "Input file was not found. FileNotFoundException : "+e.getMessage() );
		} catch (IOException e) {
			throw new BGSException( "IOException : "+e.getMessage() );
		}
	}
	private int getNbOfDirectories( int nbOfLigands ) throws BGSException{
		if ( this.bundleSize < 1 )
			throw new BGSException(" Nb of bundle size must be >= 1 ");
		int nbOfDirectories = nbOfLigands / this.bundleSize ;
		int modulo = nbOfLigands % this.bundleSize ;
		if ( modulo > 0 )
			nbOfDirectories++;
		return nbOfDirectories;
	}
	
	//Private processing methods
	private void visitDirectories( int mode ,int nbOfDirectories , int nbOfLigands ) throws BGSException {
		//String currentDirName = null;
		for ( int indexDir = 1 ; indexDir <= nbOfDirectories ; indexDir ++ ){
			this.visitSubDirectories( mode ,indexDir);
		}
	}
	private void visitSubDirectories( int mode ,int indexDir ) throws BGSException{
		int nbOfSubDir = this.bundleSize;
		if ( indexDir * this.bundleSize > nbOfLigands ){
			nbOfSubDir = this.nbOfLigands % this.bundleSize;
		}
		for ( int indexSubDir = 1; indexSubDir <= nbOfSubDir; indexSubDir++ ){
			this.recoverGoldScoreForASingleLigand( mode ,indexDir, indexSubDir );
		}
	}
	private void recoverGoldScoreForASingleLigand( int mode ,int indexDir, int indexSubDir ) throws BGSException{

		this.ligandId = this.getCurrentLigandId( indexDir , indexSubDir );
		Ligand ligand = this.getLigand( ligandId );
		Collection<Position> positions = ligand.getPositions();
		Position position = null;
		
		String[] splitLine = null;
		String sep = "[ ]+";
		String dir = this.baseDir + this.prefixDir + indexDir;
		String subDir = this.prefixSubDir + indexDir + this.suffixSubDir + indexSubDir;
		String rankFile = subDir + this.molFileRankExtension ;
		String subDirPath = dir + fileSeparator + subDir;
		String rankFilePath = subDirPath + fileSeparator + rankFile ;
		try {
			File inputFile = new File ( rankFilePath );
			if ( inputFile.exists() ){
				FileInputStream fis = new FileInputStream( inputFile );
				InputStreamReader isr = new InputStreamReader( fis );
				BufferedReader br = new BufferedReader( isr );
				
				String line = null;
				int lineNb = 0;
				int rank = 0;
				while ( (line = br.readLine()) != null ){
					lineNb++;
					splitLine = line.trim().split( sep );
					if ( splitLine != null ) {
						if ( splitLine.length == 7 ) {
							rank++;
							position = new Position( ligand );
							position.setNumber( Integer.valueOf( splitLine[0] ).intValue() );
							position.setFitness( Double.valueOf(splitLine[1]).doubleValue() );
							position.setPath( subDirPath + fileSeparator +"gold_soln_db_"+indexDir+"_m"+indexSubDir+"_"+splitLine[0]+".mol2" );
							position.setRank( rank );
							positions.add( position );
							if ( mode == KEEP_BEST ){
								break;
							}
						}
					}
				}
				br.close();
				isr.close();
				fis.close();
			} else {
				BestGoldScorer.logger.warning("The following path '"+rankFilePath+"' does not exist !");
			}
		} catch ( FileNotFoundException e){
			throw new BGSException( "Input file was not found. FileNotFoundException : "+e.getMessage() );
		} catch (IOException e) {
			throw new BGSException( "IOException : "+e.getMessage() );
		}
	}
	
	//Private util methods
	private int getCurrentLigandId( int dirNb , int subDirNb ){
		return (dirNb - 1) * this.bundleSize + subDirNb  ; 
	}
	private Ligand getLigand( int id ) throws BGSException{
		Iterator it = ligands.iterator();
		while ( it.hasNext() ){
			Ligand ligand = (Ligand) it.next();
			if ( ligand.getNumber() == id )
				return ligand;
		}
		throw new BGSException("Ligand with the following id '"+id+"' was not found.");
	}
	
	//Display Ligands
	private void shortByPositions(){
		ArrayList<Position> recordList = new ArrayList<Position>();
		Iterator it = ligands.iterator();
		while ( it.hasNext() ){
			Ligand ligand = (Ligand) it.next();
			Collection<Position> positions = ligand.getPositions();
			Iterator it2 = positions.iterator();
			while ( it2.hasNext() ){
				Position position = (Position) it2.next();
				recordList.add( position );
			}
		}
		
		//Trier les records
		Object[] objects = recordList.toArray();
		Position[] records = new Position[objects.length];
		for ( int i = 0; i < objects.length ; i ++){
			records[i] = (Position)objects[i];
		}
		Short.insertion( records , 0 , records.length-1 );
		
		this.shortedPositions = records;
		
	}
	
	
	private Position[] cutOff(){
		ArrayList<Position> cutOffPositionList = new ArrayList<Position>();
		for ( int i = 0; i < this.shortedPositions.length ; i++ ){
			if ( this.shortedPositions[i].getFitness() >= this.threshold )
				cutOffPositionList.add( this.shortedPositions[i] );
			else
				break;
		}
		Object[] objects = cutOffPositionList.toArray();
		Position[] cutOffPosition = new Position[objects.length];
		for ( int i = 0; i < objects.length ; i ++){
			cutOffPosition[i] = (Position)objects[i];
		}
		Short.insertion( cutOffPosition , 0 , cutOffPosition.length-1 );
		return cutOffPosition;
	}
	
	private void concat() throws BGSException{
		try {
			File outputScoreFile = new File( this.baseDir + this.ouputScoreFilePath );
			FileOutputStream fos = new FileOutputStream( outputScoreFile , false );
			OutputStreamWriter osw = new OutputStreamWriter( fos , "UTF-8" );
			BufferedWriter bw = new BufferedWriter( osw );
			
			File outputMol2File = new File( this.baseDir + this.outputMol2FilePath );
			FileOutputStream fos2 = new FileOutputStream( outputMol2File , false );
			OutputStreamWriter osw2 = new OutputStreamWriter( fos2 , "UTF-8" );
			BufferedWriter bw2 = new BufferedWriter( osw2 );
			
			File inputFile = null;
			FileInputStream fis = null;
			InputStreamReader isr = null;
			BufferedReader br = null;
			
			bw.write( "Name");
			bw.write("\t");
			bw.write( "GS" );
			bw.write("\t");
			bw.write( "Rank" );
			bw.newLine();
			for ( int k = 0; k < this.finalPositions.length ; k++ ){
				bw.write( this.finalPositions[k].getLigand().getName() );
				bw.write("\t");
				bw.write( this.finalPositions[k].getFitness()+"" );
				bw.write("\t");
				bw.write( this.finalPositions[k].getRank()+"" );
				bw.newLine();
				
				inputFile = new File( this.finalPositions[k].getFilePath() );
				
				if ( inputFile.exists() ){
					if ( inputFile.canRead() ){
						fis = new FileInputStream( inputFile );
						isr = new InputStreamReader( fis );
						br = new BufferedReader( isr );
						String line = null;
						while ( (line = br.readLine()) != null ){
							bw2.write( line );
							bw2.newLine();
						}
						br.close();
						isr.close();
						fis.close();
					}else {
						//Ecrire dans le fichier log pas readable
						BestGoldScorer.logger.warning("You are not allowed to read the following file '"+this.finalPositions[k].getFilePath()+"' !");
					}
				} else {
					//Ecrire dans le fichier log existe pas
					BestGoldScorer.logger.warning("The following file '"+this.finalPositions[k].getFilePath()+"' does not exist !");
				}
			}
			bw.close();
			osw.close();
			fos.close();
			bw2.close();
			osw2.close();
			fos2.close();
		} catch (FileNotFoundException e){
			throw new BGSException("FileNotFoundException : "+e.getMessage() );
		} catch (UnsupportedEncodingException e) {
			throw new BGSException("UnsupportedEncodingException : "+e.getMessage() );
		} catch (IOException e) {
			throw new BGSException("IOException : "+e.getMessage() );
		} 
	}

}
